<?php
require_once 'config.php';
require_once 'header.php';

// Redirect if not logged in
if (!isset($_SESSION['user_id'])) {
    header('Location: login.php');
    exit;
}

$user_id = $_SESSION['user_id'];

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['submit_entry'])) {
    $full_name = trim($_POST['full_name'] ?? '');
    $unit = trim($_POST['unit'] ?? '');
    $battalion = trim($_POST['battalion'] ?? '');
    $reg_no = trim($_POST['reg_no'] ?? '');
    $camp_name = trim($_POST['camp_name'] ?? '');
    $camp_date = $_POST['camp_date'] ?? '';
    $description = trim($_POST['description'] ?? '');
    $hours_attended = $_POST['hours_attended'] ?? '';
    $certificate_required = $_POST['certificate_required'] ?? 'no';

    if (empty($full_name) || empty($unit) || empty($battalion) || empty($reg_no) || empty($camp_name) || empty($camp_date) || empty($description) || empty($hours_attended)) {
        $error = 'All fields are required.';
    } elseif (!is_numeric($hours_attended) || $hours_attended <= 0) {
        $error = 'Hours attended must be a positive number.';
    } else {
        $stmt = $pdo->prepare("INSERT INTO ncc_entries (user_id, full_name, unit, battalion, reg_no, camp_name, camp_date, description, hours_attended, certificate_required) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");
        if ($stmt->execute([$user_id, $full_name, $unit, $battalion, $reg_no, $camp_name, $camp_date, $description, $hours_attended, $certificate_required])) {
            $success = 'NCC activity recorded successfully.';
        } else {
            $error = 'Submission failed. Please try again.';
        }
    }
}

// Fetch user's NCC entries
$stmt = $pdo->prepare("SELECT * FROM ncc_entries WHERE user_id = ? ORDER BY created_at DESC");
$stmt->execute([$user_id]);
$entries = $stmt->fetchAll();
?>

<h2>Welcome, <?= htmlspecialchars($_SESSION['user_name']) ?></h2>

<h3>Record New NCC Activity</h3>
<?php if (isset($error)): ?><p class="error"><?= htmlspecialchars($error) ?></p><?php endif; ?>
<?php if (isset($success)): ?><p class="success"><?= htmlspecialchars($success) ?></p><?php endif; ?>

<form method="post">
    <label for="full_name">Full Name:</label>
    <input type="text" name="full_name" id="full_name" required value="<?= htmlspecialchars($_POST['full_name'] ?? $_SESSION['user_name']) ?>">

    <label for="unit">Unit:</label>
    <input type="text" name="unit" id="unit" required value="<?= htmlspecialchars($_POST['unit'] ?? '') ?>">

    <label for="battalion">Battalion:</label>
    <input type="text" name="battalion" id="battalion" required value="<?= htmlspecialchars($_POST['battalion'] ?? '') ?>">

    <label for="reg_no">Regimental Number:</label>
    <input type="text" name="reg_no" id="reg_no" required value="<?= htmlspecialchars($_POST['reg_no'] ?? '') ?>">

    <label for="camp_name">Camp/Activity Name:</label>
    <input type="text" name="camp_name" id="camp_name" required value="<?= htmlspecialchars($_POST['camp_name'] ?? '') ?>">

    <label for="camp_date">Camp Date:</label>
    <input type="date" name="camp_date" id="camp_date" required value="<?= htmlspecialchars($_POST['camp_date'] ?? '') ?>">

    <label for="description">Description of Activities:</label>
    <textarea name="description" id="description" rows="5" required><?= htmlspecialchars($_POST['description'] ?? '') ?></textarea>

    <label for="hours_attended">Hours Attended:</label>
    <input type="number" step="0.5" name="hours_attended" id="hours_attended" required value="<?= htmlspecialchars($_POST['hours_attended'] ?? '') ?>">

    <label for="certificate_required">Certificate Required?</label>
    <select name="certificate_required" id="certificate_required">
        <option value="no" <?= (isset($_POST['certificate_required']) && $_POST['certificate_required'] == 'no') ? 'selected' : '' ?>>No</option>
        <option value="yes" <?= (isset($_POST['certificate_required']) && $_POST['certificate_required'] == 'yes') ? 'selected' : '' ?>>Yes</option>
    </select>

    <button type="submit" name="submit_entry">Submit Activity</button>
</form>

<h3>Your Past NCC Activities</h3>
<?php if (count($entries) > 0): ?>
    <table>
        <thead>
            <tr>
                <th>ID</th>
                <th>Camp Name</th>
                <th>Date</th>
                <th>Hours</th>
                <th>Certificate</th>
                <th>Status</th>
            </tr>
        </thead>
        <tbody>
        <?php foreach ($entries as $entry): ?>
            <tr>
                <td><?= $entry['id'] ?></td>
                <td><?= htmlspecialchars($entry['camp_name']) ?></td>
                <td><?= date('d M Y', strtotime($entry['camp_date'])) ?></td>
                <td><?= $entry['hours_attended'] ?></td>
                <td><?= ucfirst($entry['certificate_required']) ?></td>
                <td><?= ucfirst($entry['status']) ?></td>
            </tr>
        <?php endforeach; ?>
        </tbody>
    </table>
<?php else: ?>
    <p>You haven't recorded any NCC activities yet.</p>
<?php endif; ?>

<?php require_once 'footer.php'; ?>